/**
* \file: AlsaAudioSourceImpl.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Android Auto
*
* \author: I. Hayashi / ADITJ/SW / ihayashi@jp.adit-jv.com
*
* \copyright (c) 2013-2014 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <time.h>
#include <sys/prctl.h>
#include <adit_logging.h>
#include "AlsaAudioSourceImpl.h"

LOG_IMPORT_CONTEXT(aauto_audio)

namespace adit { namespace aauto
{
AlsaAudioSourceImpl::AlsaAudioSourceImpl(AudioSource* inSource)
{
    mSessionId= 0;
    audioSource = inSource;
    mCallbacks = nullptr;

    mStarted = false;
    mRunning = false;
}

AlsaAudioSourceImpl::~AlsaAudioSourceImpl()
{
    shutdown();
}

bool AlsaAudioSourceImpl::init()
{
    /* read configuration which was set by Application */
    if (true != getCurrConfig()) {
        LOG_ERROR((aauto_audio, "%s, init(AudioSource)  getCurrConfig() failed", mSourceName.c_str()));
        return false;
    }

    /* register callbacks to GalReceiver */
    audioSource->registerCallbacks(this);
    /* set GalReceiver::AudioSource configuration */
    audioSource->setConfig(mConfig.mSampleRate, mConfig.mNumBits, mConfig.mNumChannels);

    /* create AlsaAudio object for capture */
    mRecorder = new AlsaAudioCommon(mConfig.mAlsaLogging, mConfig.mVerbose);
    if (mRecorder == nullptr)
    {
        LOG_ERROR((aauto_audio, "%s, Error cannot create AlsaAudioCommon(source) class", mSourceName.c_str()));
        return false;
    }
    else
    {
        LOGD_DEBUG((aauto_audio, "%s, create AlsaAudioCommon(source) class successfully", mSourceName.c_str()));
    }

    return true;
}

void AlsaAudioSourceImpl::shutdown()
{
    // in case we where still running stop and notify
    if (mStarted)
    {
        mStarted = false;
        stop();
        if (mCallbacks != nullptr)
            mCallbacks->microphoneRequestCallback(false, false, false);

        LOGD_DEBUG((aauto_audio, "%s, audio source is down", mSourceName.c_str()));
    }

    // delete audio class
    mRecorder = nullptr;

    return;
}

void AlsaAudioSourceImpl::setConfigItem(string inKey, string inValue)
{
    LOGD_DEBUG((aauto_audio, "setConfigItem: inKey = %s inValue = %s", inKey.c_str(),inValue.c_str())); 
    mConfig.set(inKey, inValue);
}

void AlsaAudioSourceImpl::registerCallbacks(IAditAudioSourceCallbacks* inCallbacks)
{
    mCallbacks = inCallbacks;
}


/* IAudioSourceCallbacks */
void AlsaAudioSourceImpl::ackCallback(int sessionId, uint32_t ack)
{
    if (mSessionId == sessionId && mStarted && mConfig.mVerbose)
    {
        LOGD_VERBOSE((aauto_audio, "%s, ackCallback()  sessionId=%d, ack=%d", mSourceName.c_str(), sessionId, ack));
    }
    return;
}

int AlsaAudioSourceImpl::microphoneRequestCallback(bool open, bool ancEnabled,
        bool ecEnabled, int maxUnacked)
{
    int ret = STATUS_SUCCESS;

    LOGD_DEBUG((aauto_audio, "%s, microphoneRequestCallback() " \
            "open=%d, ancEnabled=%d, ecEnabled=%d, maxUnacked=%d",
            mSourceName.c_str(), open, ancEnabled, ecEnabled, maxUnacked));

    // start audio source
    if (true == open) {
        if (mStarted) {
            LOG_ERROR((aauto_audio, "%s, microphoneRequestCallback()  already got open!", mSourceName.c_str()));
            return -1;
        }

        mStarted = true;
        bool callbackOK = false;

        if(mCallbacks != nullptr) {
            /* notify Application to start audio capture */
            ret = mCallbacks->microphoneRequestCallback(open, ancEnabled, ecEnabled);
            if (STATUS_SUCCESS == ret) {
                callbackOK = true;
            } else {
                LOG_ERROR((aauto_audio, "%s, microphoneRequestCallback()  from upper layer failed with ret=%d",
                        mSourceName.c_str(), ret));
            }
        }
        if (callbackOK)
        {
            /* read and set configuration again because Application
             * got set new configuration values due to microphoneRequestCallback(open==true) */
            if (true != getCurrConfig()) {
                LOG_ERROR((aauto_audio, "%s, microphoneRequestCallback()  getCurrConfig() failed", mSourceName.c_str()));
                ret = -1;
            } else {
                /* set GalReceiver::AudioSource configuration */
                audioSource->setConfig(mConfig.mSampleRate, mConfig.mNumBits, mConfig.mNumChannels);

                /* prepare and start audio capture */
                if(true != start(maxUnacked)) {
                    LOG_ERROR((aauto_audio, "%s, microphoneRequestCallback()  start() failed!", mSourceName.c_str()));
                    ret = -1;
                }
            }
        }
    } else { // stop audio source
        if (!mStarted) {
            LOG_ERROR((aauto_audio, "%s, microphoneRequestCallback()  stop without start", mSourceName.c_str()));
            // continue anyway
        }
        mStarted = false;

        /* stop audio capture */
        stop();

        if(mCallbacks != nullptr) {
            /* notify Application to stop audio capture */
            ret = mCallbacks->microphoneRequestCallback(open, ancEnabled, ecEnabled);
        }
    }

    return mSessionId; /* This meaning sessionId = 0 */
}


uint64_t AlsaAudioSourceImpl::MicrophoneReaderThread::getTimestamp()
{
    struct timespec tp;
    clock_gettime(CLOCK_MONOTONIC, &tp);
    return (tp.tv_sec * NSEC_PER_SEC + tp.tv_nsec) / 1000;
}

void AlsaAudioSourceImpl::MicrophoneReaderThread::run()
{
    /* time to sleep in case ReadWrite() return with unrecoverable error  */
    uint32_t waitTimeMs = MICROPHONE_WORTH_OF_SAMPLE_MS;

    int err = 0;
    /* Create IoBuffer with period aligned size (8ms, 16ms) to read from ALSA device */
    shared_ptr<IoBuffer> data(new IoBuffer(mAudioSource->mRecorder->GetIdealPeriodBytes()));
    LOGD_DEBUG((aauto_audio, "%s, MicrophoneReaderThread started.", mAudioSource->mSourceName.c_str()));

    /* set thread name */
    prctl(PR_SET_NAME, "AlsaAudioSource", 0, 0, 0);

    while (mAudioSource->mRunning)
    {
        /* read audio data from AlsaAudio object (ALSA device) into IoBuffer */
        if ((err = mAudioSource->mRecorder->ReadWrite((uint8_t*)data->raw(), data->size())) < 0)
        {
            LOG_ERROR((aauto_audio, "%s, Failed to read from microphone err=%d. Wait for %u ms.",
                    mAudioSource->mSourceName.c_str(), err,  waitTimeMs));
            if (mAudioSource->mCallbacks != nullptr)
            {
                mAudioSource->mCallbacks->notifyErrorCallback(AUDIO_SOURCE_READ_ERROR);
            }
            /* in case of error, (wait)run until stop received */
            usleep(waitTimeMs * 1000);
            /* flush data buffer */
            memset(data->raw(), 0, data->size());
        }
        /* send capture to MD */
        mAudioSource->audioSource->send(getTimestamp(), data);
    }
    LOGD_DEBUG((aauto_audio, "%s, MicrophoneReaderThread exit.",
            mAudioSource->mSourceName.c_str()));
}

/* private methods */
bool AlsaAudioSourceImpl::getCurrConfig()
{
    mSourceName = "AudioSourceMic";

    // read out configuration parameters
    if (true != mConfig.ResultConfig()) {
        LOG_ERROR((aauto_audio, "getCurrConfig()  ResultConfig() failed"));
        return false;
    }

    if ((mConfig.mSampleRate == MICROPHONE_SAMPLING_RATE) && 
        (mConfig.mNumBits == MICROPHONE_BITS_PER_SAMPLE) && 
        (mConfig.mNumChannels == MICROPHONE_CHANNELS))
    {
        /* ok */
    }
    else
    {
        LOG_ERROR((aauto_audio, "%s, Error setting parameter sampling-rate, bits-per-sample, channels",
                mSourceName.c_str()));
        return false;
    }

    return true;
}

bool AlsaAudioSourceImpl::start(int maxUnacked)
{
    /* set-up ALSA device for capture */
    if (prepare())
    {
        LOGD_DEBUG((aauto_audio, "%s, ALSA prepare successfully", mSourceName.c_str()));
    }
    else
    {
        LOG_ERROR((aauto_audio, "%s, Error ALSA prepare", mSourceName.c_str()));
        return false;
    }

    mSem = new Semaphore(maxUnacked);
    mReaderThread = new MicrophoneReaderThread(this);
    mRunning = true;
    /* start MicrophoneReaderThread to capture audio */
    mReaderThread->start();
    if (!mConfig.mDisablePrio)
    {
        mReaderThread->setPriority(mConfig.mThreadPrio);
    }

    return true;
}

void AlsaAudioSourceImpl::stop()
{
    mRunning = false;

    if (mRecorder != nullptr) {
        /* stop processing as soon as possible */
        mRecorder->setAlsaAudioState(mRecorder->AlsaAudioStates::AlsaAudioStop);
    }
    /* close audio source thread and wait for closure */
    if (mReaderThread != nullptr)
    {
        mReaderThread->join();
        mReaderThread = nullptr;
    }

    /* stop audio capture */
    if (mRecorder != nullptr)
    {
        mRecorder->StopStreaming();
        mRecorder->ReleasePCM();
    }
}

/* Preparations of ALSA */
bool AlsaAudioSourceImpl::prepare()
{
    LOGD_DEBUG((aauto_audio, "%s, audio stream start (sample rate: %d, bits per channel: %d, channels: %d)",
            mSourceName.c_str(), mConfig.mSampleRate, mConfig.mNumBits, mConfig.mNumChannels));

    mRecorder->SetStreamName(mSourceName.c_str());
    mRecorder->SetDir( SND_PCM_STREAM_CAPTURE );
    mRecorder->SetRate( mConfig.mSampleRate );
    mRecorder->SetInitTout(mConfig.mInitToutms);
    mRecorder->SetChannels( mConfig.mNumChannels );
    mRecorder->SetIdealPeriodMs( mConfig.mPeriodms );
    if (mConfig.mBufferperiods >= 0)
    {
        mRecorder->SetBufferIdealPeriods( mConfig.mBufferperiods );
    }
    if (mConfig.mSilencems >= 0)
    {
        mRecorder->SetPrefillMs ( mConfig.mSilencems );
    }
    mRecorder->SetFormat( SND_PCM_FORMAT_S16_LE );

    if (0 != mRecorder->SetupPCM((char*)mConfig.mDevice.c_str())) {
        return false;
    }

    LOGD_DEBUG((aauto_audio, "%s, alsa audio source : device=%s, period=%dms",
            mSourceName.c_str(), mConfig.mDevice.c_str(), mConfig.mPeriodms));

    return true;
}

} } /* namespace adit { namespace aauto */
